'use client'

import { useState, useEffect } from 'react'
import { useRouter, useParams } from 'next/navigation'
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card'
import { Button } from '@/components/ui/button'
import { Input } from '@/components/ui/input'
import { Label } from '@/components/ui/label'
import {
  BookOpen,
  Save,
  ArrowLeft,
  Eye,
  Bold,
  Italic,
  Heading1,
  Heading2,
  Heading3,
  Heading4,
  Image,
  List,
  Code,
  Link as LinkIcon
} from 'lucide-react'
import Link from 'next/link'

export default function EditBlogPage() {
  const router = useRouter()
  const params = useParams()
  const [loading, setLoading] = useState(true)
  const [saving, setSaving] = useState(false)
  const [preview, setPreview] = useState(false)
  
  const [formData, setFormData] = useState({
    title: '',
    slug: '',
    excerpt: '',
    content: '',
    category: 'Genel',
    coverImage: '',
    author: '',
    published: false
  })

  const categories = ['Stratejiler', 'Taktikler', 'Analizler', 'Yönetim', 'Eğitim', 'Haberler', 'Genel']

  useEffect(() => {
    loadPost()
  }, [])

  const loadPost = async () => {
    try {
      const res = await fetch(`/api/admin/blog/${params.id}`)
      if (res.ok) {
        const post = await res.json()
        setFormData({
          title: post.title,
          slug: post.slug,
          excerpt: post.excerpt,
          content: post.content,
          category: post.category,
          coverImage: post.coverImage || '',
          author: post.author,
          published: post.published
        })
      } else {
        alert('Blog yazısı bulunamadı')
        router.push('/admin/blog')
      }
    } catch (error) {
      console.error('Blog yazısı yüklenirken hata:', error)
    } finally {
      setLoading(false)
    }
  }

  const insertMarkdown = (type: string) => {
    const textarea = document.getElementById('content') as HTMLTextAreaElement
    if (!textarea) return

    const start = textarea.selectionStart
    const end = textarea.selectionEnd
    const selectedText = formData.content.substring(start, end)
    let newText = formData.content

    switch (type) {
      case 'h1':
        newText = formData.content.substring(0, start) + '# ' + selectedText + '\n' + formData.content.substring(end)
        break
      case 'h2':
        newText = formData.content.substring(0, start) + '## ' + selectedText + '\n' + formData.content.substring(end)
        break
      case 'h3':
        newText = formData.content.substring(0, start) + '### ' + selectedText + '\n' + formData.content.substring(end)
        break
      case 'h4':
        newText = formData.content.substring(0, start) + '#### ' + selectedText + '\n' + formData.content.substring(end)
        break
      case 'bold':
        newText = formData.content.substring(0, start) + '**' + selectedText + '**' + formData.content.substring(end)
        break
      case 'italic':
        newText = formData.content.substring(0, start) + '*' + selectedText + '*' + formData.content.substring(end)
        break
      case 'image':
        const imageUrl = prompt('Resim URL\'sini girin:')
        if (imageUrl) {
          newText = formData.content.substring(0, start) + `![${selectedText || 'Resim'}](${imageUrl})` + formData.content.substring(end)
        }
        break
      case 'link':
        const linkUrl = prompt('Link URL\'sini girin:')
        if (linkUrl) {
          newText = formData.content.substring(0, start) + `[${selectedText || 'Link'}](${linkUrl})` + formData.content.substring(end)
        }
        break
      case 'list':
        newText = formData.content.substring(0, start) + '- ' + selectedText + '\n' + formData.content.substring(end)
        break
      case 'code':
        newText = formData.content.substring(0, start) + '```\n' + selectedText + '\n```' + formData.content.substring(end)
        break
    }

    setFormData({ ...formData, content: newText })
    setTimeout(() => {
      textarea.focus()
      textarea.selectionStart = start
      textarea.selectionEnd = start
    }, 0)
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    
    if (!formData.title || !formData.slug || !formData.excerpt || !formData.content) {
      alert('Lütfen tüm zorunlu alanları doldurun')
      return
    }

    setSaving(true)
    try {
      const res = await fetch(`/api/admin/blog/${params.id}`, {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(formData)
      })

      if (res.ok) {
        alert('Blog yazısı başarıyla güncellendi!')
        router.push('/admin/blog')
      } else {
        const data = await res.json()
        alert(data.error || 'Bir hata oluştu')
      }
    } catch (error) {
      console.error('Blog yazısı güncellenirken hata:', error)
      alert('Bir hata oluştu')
    } finally {
      setSaving(false)
    }
  }

  const renderMarkdown = (text: string) => {
    return text
      .replace(/^# (.*$)/gim, '<h1 class="text-4xl font-bold mb-4">$1</h1>')
      .replace(/^## (.*$)/gim, '<h2 class="text-3xl font-bold mb-3">$1</h2>')
      .replace(/^### (.*$)/gim, '<h3 class="text-2xl font-bold mb-2">$1</h3>')
      .replace(/^#### (.*$)/gim, '<h4 class="text-xl font-bold mb-2">$1</h4>')
      .replace(/\*\*(.*?)\*\*/gim, '<strong>$1</strong>')
      .replace(/\*(.*?)\*/gim, '<em>$1</em>')
      .replace(/!\[(.*?)\]\((.*?)\)/gim, '<img src="$2" alt="$1" class="max-w-full h-auto rounded-lg my-4" />')
      .replace(/\[(.*?)\]\((.*?)\)/gim, '<a href="$2" class="text-green-400 hover:underline">$1</a>')
      .replace(/^- (.*$)/gim, '<li>$1</li>')
      .replace(/```([\s\S]*?)```/gim, '<pre class="bg-black/30 p-4 rounded-lg my-4 overflow-x-auto"><code>$1</code></pre>')
      .split('\n').map(line => line.startsWith('<li>') ? line : `<p class="mb-2">${line}</p>`).join('\n')
  }

  if (loading) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div className="inline-block h-12 w-12 animate-spin rounded-full border-4 border-solid border-green-400 border-r-transparent"></div>
          <p className="mt-4 text-foreground/60">Yükleniyor...</p>
        </div>
      </div>
    )
  }

  return (
    <div className="p-8 space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-3">
          <Link href="/admin/blog">
            <Button variant="ghost" size="sm">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Geri
            </Button>
          </Link>
          <BookOpen className="h-8 w-8 text-green-400" />
          <div>
            <h1 className="text-3xl font-bold">Blog Yazısını Düzenle</h1>
            <p className="text-foreground/60">Blog yazısını güncelle</p>
          </div>
        </div>
        <div className="flex items-center space-x-2">
          <Button
            variant="outline"
            onClick={() => setPreview(!preview)}
            className="border-purple-500/30 text-purple-400"
          >
            <Eye className="h-4 w-4 mr-2" />
            {preview ? 'Düzenle' : 'Önizle'}
          </Button>
        </div>
      </div>

      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-6">
            <Card className="glass-dark border-white/10">
              <CardHeader>
                <CardTitle>İçerik</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="title">Başlık *</Label>
                  <Input
                    id="title"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    placeholder="Blog yazısı başlığı"
                    className="glass border-white/10 focus:border-green-500/50"
                    required
                  />
                </div>

                <div>
                  <Label htmlFor="slug">Slug (URL) *</Label>
                  <Input
                    id="slug"
                    value={formData.slug}
                    onChange={(e) => setFormData({ ...formData, slug: e.target.value })}
                    placeholder="blog-yazisi-url"
                    className="glass border-white/10 focus:border-green-500/50"
                    required
                  />
                  <p className="text-xs text-foreground/60 mt-1">URL: /blog/{formData.slug}</p>
                </div>

                <div>
                  <Label htmlFor="excerpt">Özet *</Label>
                  <textarea
                    id="excerpt"
                    value={formData.excerpt}
                    onChange={(e) => setFormData({ ...formData, excerpt: e.target.value })}
                    placeholder="Kısa özet (maksimum 200 karakter)"
                    className="w-full min-h-[100px] glass border border-white/10 focus:border-green-500/50 rounded-lg p-3 bg-black/20"
                    maxLength={200}
                    required
                  />
                  <p className="text-xs text-foreground/60 mt-1">{formData.excerpt.length}/200</p>
                </div>

                {!preview && (
                  <>
                    <div>
                      <Label>Formatla</Label>
                      <div className="flex flex-wrap gap-2 mt-2">
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('h1')} className="border-white/10">
                          <Heading1 className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('h2')} className="border-white/10">
                          <Heading2 className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('h3')} className="border-white/10">
                          <Heading3 className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('h4')} className="border-white/10">
                          <Heading4 className="h-4 w-4" />
                        </Button>
                        <div className="w-px h-6 bg-white/10"></div>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('bold')} className="border-white/10">
                          <Bold className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('italic')} className="border-white/10">
                          <Italic className="h-4 w-4" />
                        </Button>
                        <div className="w-px h-6 bg-white/10"></div>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('image')} className="border-white/10">
                          <Image className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('link')} className="border-white/10">
                          <LinkIcon className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('list')} className="border-white/10">
                          <List className="h-4 w-4" />
                        </Button>
                        <Button type="button" size="sm" variant="outline" onClick={() => insertMarkdown('code')} className="border-white/10">
                          <Code className="h-4 w-4" />
                        </Button>
                      </div>
                    </div>

                    <div>
                      <Label htmlFor="content">İçerik (Markdown) *</Label>
                      <textarea
                        id="content"
                        value={formData.content}
                        onChange={(e) => setFormData({ ...formData, content: e.target.value })}
                        placeholder="Blog yazısı içeriği (Markdown formatında)"
                        className="w-full min-h-[500px] glass border border-white/10 focus:border-green-500/50 rounded-lg p-4 bg-black/20 font-mono text-sm"
                        required
                      />
                    </div>
                  </>
                )}

                {preview && (
                  <div className="glass-dark p-6 rounded-lg border border-white/10">
                    <h2 className="text-3xl font-bold mb-4">{formData.title}</h2>
                    <div className="prose prose-invert max-w-none" dangerouslySetInnerHTML={{ __html: renderMarkdown(formData.content) }} />
                  </div>
                )}
              </CardContent>
            </Card>
          </div>

          {/* Sidebar */}
          <div className="space-y-6">
            <Card className="glass-dark border-white/10">
              <CardHeader>
                <CardTitle>Ayarlar</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div>
                  <Label htmlFor="category">Kategori *</Label>
                  <select
                    id="category"
                    value={formData.category}
                    onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                    className="w-full glass border border-white/10 focus:border-green-500/50 rounded-lg p-3 bg-black/20"
                  >
                    {categories.map((cat) => (
                      <option key={cat} value={cat}>{cat}</option>
                    ))}
                  </select>
                </div>

                <div>
                  <Label htmlFor="author">Yazar</Label>
                  <Input
                    id="author"
                    value={formData.author}
                    onChange={(e) => setFormData({ ...formData, author: e.target.value })}
                    placeholder="Admin"
                    className="glass border-white/10 focus:border-green-500/50"
                  />
                </div>

                <div>
                  <Label htmlFor="coverImage">Kapak Resmi URL</Label>
                  <Input
                    id="coverImage"
                    value={formData.coverImage}
                    onChange={(e) => setFormData({ ...formData, coverImage: e.target.value })}
                    placeholder="https://example.com/image.jpg"
                    className="glass border-white/10 focus:border-green-500/50"
                  />
                </div>

                <div className="flex items-center space-x-2">
                  <input
                    type="checkbox"
                    id="published"
                    checked={formData.published}
                    onChange={(e) => setFormData({ ...formData, published: e.target.checked })}
                    className="h-4 w-4"
                  />
                  <Label htmlFor="published" className="cursor-pointer">Yayınla</Label>
                </div>
              </CardContent>
            </Card>

            <Button
              type="submit"
              disabled={saving}
              className="w-full bg-gradient-to-r from-green-500 to-yellow-400 hover:from-green-600 hover:to-yellow-500 text-black font-bold h-12"
            >
              {saving ? (
                <>
                  <div className="h-4 w-4 animate-spin rounded-full border-2 border-black border-t-transparent mr-2"></div>
                  Kaydediliyor...
                </>
              ) : (
                <>
                  <Save className="h-4 w-4 mr-2" />
                  Güncelle
                </>
              )}
            </Button>
          </div>
        </div>
      </form>
    </div>
  )
}



